// Enhanced Admin JavaScript
class AdminPanel {
    constructor() {
        this.init();
    }

    init() {
        this.setupDeleteConfirmation();
        this.setupFormValidation();
        this.setupAutoHideAlerts();
        this.setupImagePreview();
        this.setupSidebarToggle();
        this.setupTableSorting();
        this.setupSearchFilters();
        this.setupLoadingStates();
        this.setupNotifications();
        this.setupKeyboardShortcuts();
    }

    // Enhanced delete confirmation with better UX
    setupDeleteConfirmation() {
        document.querySelectorAll('.btn-delete').forEach(button => {
            button.addEventListener('click', function(e) {
                e.preventDefault();
                
                const itemName = this.dataset.itemName || 'data ini';
                const confirmModal = document.createElement('div');
                confirmModal.className = 'modal fade';
                confirmModal.innerHTML = `
                    <div class="modal-dialog modal-dialog-centered">
                        <div class="modal-content">
                            <div class="modal-header border-0">
                                <h5 class="modal-title">
                                    <i class="bi bi-exclamation-triangle text-warning me-2"></i>
                                    Konfirmasi Hapus
                                </h5>
                            </div>
                            <div class="modal-body">
                                <p>Apakah Anda yakin ingin menghapus <strong>${itemName}</strong>?</p>
                                <p class="text-muted small">Tindakan ini tidak dapat dibatalkan.</p>
                            </div>
                            <div class="modal-footer border-0">
                                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Batal</button>
                                <button type="button" class="btn btn-danger" id="confirmDelete">
                                    <i class="bi bi-trash me-1"></i>Hapus
                                </button>
                            </div>
                        </div>
                    </div>
                `;
                
                document.body.appendChild(confirmModal);
                const modal = new bootstrap.Modal(confirmModal);
                modal.show();
                
                confirmModal.querySelector('#confirmDelete').addEventListener('click', () => {
                    if (this.href) {
                        window.location.href = this.href;
                    } else if (this.closest('form')) {
                        this.closest('form').submit();
                    }
                });
                
                confirmModal.addEventListener('hidden.bs.modal', () => {
                    document.body.removeChild(confirmModal);
                });
            });
        });
    }

    // Enhanced form validation
    setupFormValidation() {
        const forms = document.querySelectorAll('.needs-validation');
        Array.from(forms).forEach(form => {
            form.addEventListener('submit', event => {
                if (!form.checkValidity()) {
                    event.preventDefault();
                    event.stopPropagation();
                    
                    // Focus on first invalid field
                    const firstInvalid = form.querySelector(':invalid');
                    if (firstInvalid) {
                        firstInvalid.focus();
                        firstInvalid.scrollIntoView({ behavior: 'smooth', block: 'center' });
                    }
                }
                form.classList.add('was-validated');
            }, false);
            
            // Real-time validation
            const inputs = form.querySelectorAll('input, select, textarea');
            inputs.forEach(input => {
                input.addEventListener('blur', () => {
                    if (form.classList.contains('was-validated')) {
                        input.classList.toggle('is-valid', input.checkValidity());
                        input.classList.toggle('is-invalid', !input.checkValidity());
                    }
                });
            });
        });
    }

    // Enhanced auto-hide alerts with animation
    setupAutoHideAlerts() {
        const alerts = document.querySelectorAll('.alert');
        alerts.forEach(alert => {
            // Add close button if not exists
            if (!alert.querySelector('.btn-close')) {
                const closeBtn = document.createElement('button');
                closeBtn.className = 'btn-close';
                closeBtn.setAttribute('data-bs-dismiss', 'alert');
                alert.appendChild(closeBtn);
            }
            
            // Auto-hide after 5 seconds
            setTimeout(() => {
                if (alert.parentNode) {
                    alert.style.opacity = '0';
                    alert.style.transform = 'translateY(-20px)';
                    setTimeout(() => {
                        if (alert.parentNode) {
                            alert.remove();
                        }
                    }, 300);
                }
            }, 5000);
        });
    }

    // Enhanced image preview with multiple images support
    setupImagePreview() {
        document.querySelectorAll('input[type="file"][accept*="image"]').forEach(input => {
            input.addEventListener('change', function(e) {
                const files = Array.from(e.target.files);
                const previewContainer = document.getElementById('image-preview-container') || 
                                       this.parentNode.querySelector('.image-preview-container');
                
                if (previewContainer) {
                    previewContainer.innerHTML = '';
                    
                    files.forEach((file, index) => {
                        if (file.type.startsWith('image/')) {
                            const reader = new FileReader();
                            reader.onload = function(e) {
                                const imgWrapper = document.createElement('div');
                                imgWrapper.className = 'position-relative d-inline-block me-2 mb-2';
                                imgWrapper.innerHTML = `
                                    <img src="${e.target.result}" class="img-thumbnail" style="width: 100px; height: 100px; object-fit: cover;">
                                    <button type="button" class="btn btn-danger btn-sm position-absolute top-0 end-0 rounded-circle" 
                                            style="width: 25px; height: 25px; padding: 0;" onclick="this.parentNode.remove()">
                                        <i class="bi bi-x"></i>
                                    </button>
                                `;
                                previewContainer.appendChild(imgWrapper);
                            };
                            reader.readAsDataURL(file);
                        }
                    });
                }
            });
        });
    }

    // Mobile sidebar toggle
    setupSidebarToggle() {
        // Create toggle button if not exists
        if (!document.querySelector('.sidebar-toggle')) {
            const toggleBtn = document.createElement('button');
            toggleBtn.className = 'sidebar-toggle btn';
            toggleBtn.innerHTML = '<i class="bi bi-list"></i>';
            document.body.appendChild(toggleBtn);
            
            // Create overlay
            const overlay = document.createElement('div');
            overlay.className = 'sidebar-overlay';
            document.body.appendChild(overlay);
            
            const sidebar = document.querySelector('.sidebar');
            
            toggleBtn.addEventListener('click', () => {
                sidebar.classList.toggle('show');
                overlay.classList.toggle('show');
            });
            
            overlay.addEventListener('click', () => {
                sidebar.classList.remove('show');
                overlay.classList.remove('show');
            });
        }
    }

    // Table sorting functionality
    setupTableSorting() {
        document.querySelectorAll('th[data-sort]').forEach(th => {
            th.style.cursor = 'pointer';
            th.addEventListener('click', () => {
                const table = th.closest('table');
                const tbody = table.querySelector('tbody');
                const rows = Array.from(tbody.querySelectorAll('tr'));
                const column = th.cellIndex;
                const isAsc = th.classList.contains('sort-asc');
                
                // Remove existing sort classes
                table.querySelectorAll('th').forEach(header => {
                    header.classList.remove('sort-asc', 'sort-desc');
                });
                
                // Add new sort class
                th.classList.add(isAsc ? 'sort-desc' : 'sort-asc');
                
                // Sort rows
                rows.sort((a, b) => {
                    const aVal = a.cells[column].textContent.trim();
                    const bVal = b.cells[column].textContent.trim();
                    
                    if (isAsc) {
                        return bVal.localeCompare(aVal, undefined, { numeric: true });
                    } else {
                        return aVal.localeCompare(bVal, undefined, { numeric: true });
                    }
                });
                
                // Reorder DOM
                rows.forEach(row => tbody.appendChild(row));
            });
        });
    }

    // Search and filter functionality
    setupSearchFilters() {
        const searchInputs = document.querySelectorAll('[data-search-target]');
        searchInputs.forEach(input => {
            const targetSelector = input.dataset.searchTarget;
            const targets = document.querySelectorAll(targetSelector);
            
            input.addEventListener('input', function() {
                const query = this.value.toLowerCase();
                
                targets.forEach(target => {
                    const text = target.textContent.toLowerCase();
                    const shouldShow = text.includes(query);
                    target.style.display = shouldShow ? '' : 'none';
                });
            });
        });
    }

    // Loading states for forms and buttons
    setupLoadingStates() {
        document.querySelectorAll('form').forEach(form => {
            form.addEventListener('submit', function() {
                const submitBtn = this.querySelector('button[type="submit"]');
                if (submitBtn && !submitBtn.disabled) {
                    const originalText = submitBtn.innerHTML;
                    submitBtn.innerHTML = '<span class="spinner-border spinner-border-sm me-2"></span>Memproses...';
                    submitBtn.disabled = true;
                    
                    // Re-enable after timeout (fallback)
                    setTimeout(() => {
                        submitBtn.innerHTML = originalText;
                        submitBtn.disabled = false;
                    }, 10000);
                }
            });
        });
    }

    // Notification system
    setupNotifications() {
        window.showNotification = (message, type = 'success', duration = 3000) => {
            const notification = document.createElement('div');
            notification.className = `notification ${type}`;
            notification.textContent = message;
            
            document.body.appendChild(notification);
            
            setTimeout(() => {
                notification.style.opacity = '0';
                notification.style.transform = 'translateX(100%)';
                setTimeout(() => {
                    if (notification.parentNode) {
                        notification.remove();
                    }
                }, 300);
            }, duration);
        };
    }

    // Keyboard shortcuts
    setupKeyboardShortcuts() {
        document.addEventListener('keydown', (e) => {
            // Ctrl/Cmd + S to save form
            if ((e.ctrlKey || e.metaKey) && e.key === 's') {
                e.preventDefault();
                const form = document.querySelector('form');
                if (form) {
                    form.submit();
                }
            }
            
            // Escape to close modals
            if (e.key === 'Escape') {
                const openModal = document.querySelector('.modal.show');
                if (openModal) {
                    bootstrap.Modal.getInstance(openModal).hide();
                }
            }
        });
    }
}

// Initialize admin panel when DOM is loaded
document.addEventListener('DOMContentLoaded', () => {
    new AdminPanel();
    
    // Fix for form elements clickability
    fixFormClickability();
});

// Fix form elements clickability issues
function fixFormClickability() {
    // Ensure all form elements are clickable
    const formElements = document.querySelectorAll('.admin-card input, .admin-card select, .admin-card textarea, .admin-card button');
    
    formElements.forEach(element => {
        // Force pointer events
        element.style.pointerEvents = 'auto';
        element.style.position = 'relative';
        element.style.zIndex = '100';
        
        // Add click event listener to ensure responsiveness
        if (element.tagName === 'BUTTON' || element.type === 'submit') {
            element.style.cursor = 'pointer';
        }
        
        // Fix for input focus
        if (element.tagName === 'INPUT' || element.tagName === 'TEXTAREA' || element.tagName === 'SELECT') {
            element.addEventListener('mousedown', function(e) {
                e.stopPropagation();
            });
            
            element.addEventListener('click', function(e) {
                e.stopPropagation();
                this.focus();
            });
        }
    });
    
    // Fix for labels
    const labels = document.querySelectorAll('.admin-card .form-label');
    labels.forEach(label => {
        label.style.pointerEvents = 'auto';
        label.style.cursor = 'pointer';
        label.style.zIndex = '101';
        
        label.addEventListener('click', function(e) {
            e.stopPropagation();
            const forAttr = this.getAttribute('for');
            if (forAttr) {
                const targetInput = document.getElementById(forAttr);
                if (targetInput) {
                    targetInput.focus();
                }
            }
        });
    });
    
    // Fix for form containers
    const adminCards = document.querySelectorAll('.admin-card');
    adminCards.forEach(card => {
        // Remove any hover transforms that might interfere
        card.addEventListener('mouseenter', function() {
            this.style.transform = 'none';
        });
        
        // Ensure form is clickable
        const form = card.querySelector('form');
        if (form) {
            form.style.pointerEvents = 'auto';
            form.style.position = 'relative';
            form.style.zIndex = '50';
        }
    });
    
    // Fix for file inputs specifically
    const fileInputs = document.querySelectorAll('input[type="file"]');
    fileInputs.forEach(input => {
        input.style.pointerEvents = 'auto';
        input.style.cursor = 'pointer';
        input.style.zIndex = '200';
        input.style.position = 'relative';
    });
    
    // Fix for submit buttons
    const submitButtons = document.querySelectorAll('button[type="submit"], .btn-primary');
    submitButtons.forEach(button => {
        button.style.pointerEvents = 'auto';
        button.style.cursor = 'pointer';
        button.style.zIndex = '999';
        button.style.position = 'relative';
        button.style.display = 'inline-block';
        
        // Ensure button click works
        button.addEventListener('click', function(e) {
            e.stopPropagation();
            if (this.type === 'submit' && this.form) {
                // Let the form handle submission
                return true;
            }
        });
    });
    
    console.log('Form clickability fixes applied');
}

// Simple button loading state - no double loading
function setButtonLoading(button, isLoading = true) {
    if (isLoading) {
        button.disabled = true;
        const originalText = button.innerHTML;
        button.setAttribute('data-original-text', originalText);
        button.innerHTML = '<span class="spinner-border spinner-border-sm me-2"></span>Tunggu sebentar yahh..';
    } else {
        button.disabled = false;
        const originalText = button.getAttribute('data-original-text');
        if (originalText) {
            button.innerHTML = originalText;
            button.removeAttribute('data-original-text');
        }
    }
}